classdef DPP_simulator < handle
    
    % DEDEBUG: look for this to find questions
    
    properties
        
        % file names under which the results wiull be saved
        a = '-------------      save results paths       -------------';
        file_name_sim_mat
        file_name_torque_map_img
        file_name_nucleus_run_img
        file_name_state_history_img
        file_name_split_cell_img
        
         % flags for automatic saving of some results
        b = '--------  flags for automatic saving of results  --------';
        auto_save_nucleus_run
        auto_save_torque_map 
        auto_save_state_history
        auto_save_split_image
             
        c = '-------------      torque map parameters      -----------';
        torque_map              % holder for the torque map
        torque_map_angle_step   % angle step for the torque map    
        auto_calc_torque_map    % flag for automatic calculation of the torque map

        d = '-------------      Simulation parameters      -----------';

        angle_btw_mt_deg        % how dense microtubules are: 1 MT per indicated number of degrees        
        duration_full           % number of iterations to run
        duration_equilibrium    % number of last iterations whithin which the states are checked if they are around equilibrium.        
        jump_chance             % prob to jump?
        inertia                 % threshold for re_roll        
        show_log
        show_visually
        
        current_cell
       
    end
    
    
    properties (Access = private)
        XcYc            % {[x(:) y(:)], ..., [x(:), y(:)]} - cell array of n*2 matrices ([x(:), y(:)]). 
                        % pixels belonging to the cell in each z-slice; 
                        % Example: XcYc{25} contains [x,y] of the 2D cell slice at z = 25.
               
        font_size       % font size for images
        message_str
        
        % parameter for splitting the cells according to the final spindle
        % position. threshold distance from the division plane:
        split_distance_threshold
        
        echec           % state of mistakes. Basically measures whether to GO ON or NOT
        init            % a state for initialization of some parameters. 0 during normal run, 2 and 1 during intitializaiton
        
        cur_iter           % current iteration N 
        
         % varied parameters [x, y, z, theta, phi] are varied 
        % directly inside the  "current_cell"; these are their copies that
        % keep in memory the previous state:
        xsave               % memorized x position of the nucleus
        ysave               % memorized y position of the nucleus
        zsave               % memorized z position of the nucleus
        thetasave           % memorized theta orientation of the spindle
        phisave             % memorized pni   orientation of the spindle 
        
        re_roll_bool        % "delance" form french "rethrow"; bool. Switcher 1/0 for rethrowing "de" or not
        re_roll_value       % if re_roll=1 (rethrowting ON): chooses randomly what parameter out of ...
                            % [theta(1,2), phi(3,4), x(5,6), y(7,8), z(9,10)] to change and how to change (+1/-1). 
                            
        % Force state
        tr_fr_state_current % current    force state: [theta_torque, phi_torque, force_x, force_y, force_z]
        tr_fr_state_new     % new        force state: [theta_torque, phi_torque, force_x, force_y, force_z]
        maxpot              % minimal    force state: [theta_torque, phi_torque, force_x, force_y, force_z]
        minpot              % maximal    force state: [theta_torque, phi_torque, force_x, force_y, force_z]            
        tr_fr_history           % history of force states, matrix n*5: [theta_torque, phi_torque, force_x, force_y, force_z]
        it                  % iterator for accepted states. Different from "ittot", because NOT ALL states are calculated.
        
         XYZ                 % FULL history of x,y,z.      Matrix of n*3: [x(:), y(:), z(:)], where n is current iteration number
        ThPhi           	% FULL history of theta, phi. Matrix of n*2: [theta(:), phi(:)], where n is current iteration number 
        choix               % FULL history of accepted/rejected states. If 1: state was favorable. 0 = rejected state (unfavorable), 2 = jump.
        
        xyz_th_phi_history      % history of states x,y,z,theta, phi THAT WERE USED for force calculation. Matrix of n*5. 

        torque_map_theta          
        torque_map_phi
        noLog_each_N_iteration
        plot_jumps = 0
    end
    
    
    
    
    methods 
        
        function obj = DPP_simulator
            
            % INITIATE PARAMETERS
            % iteration parameters
            obj.init                        = 2;                        % switcher for initialization; responsible for first 2 iterations.
            obj.cur_iter                    = 0;                        % current iteration numbner
            obj.echec                       = 0;                        % error check parameter
            obj.duration_full               = 3000;                     % maximal number of iterations
            obj.duration_equilibrium        = 600;                      % number of last iterations whithin which the states are checked if they are around equilibrium.

            % Parameters responsible for random varying of parameters [x,y,z,th,phi]
            obj.re_roll_bool                = 1;                        % if 1 -> re-roll for new changing variable (de);
            obj.re_roll_value               = 0;                        % re_roll_value - what and how to re-roll.  re_roll_value = ceil(10.*rand(1,1)); [1:10]
            obj.inertia                     = 80;                       % inertia of the system: probability to keep the same direciton of varying a parameter.  
            obj.jump_chance                 = 20;                       % probability of jump from an equilibrium state. in percents, 10 = 10% of hance of jumping

            % Force states
            obj.tr_fr_state_current         = ones(1, 5);               % previous   force state: [theta_torque, phi_torque, force_x, force_y, force_z]
            obj.tr_fr_state_new             = NaN(1, 5);                % current    force state: [theta_torque, phi_torque, force_x, force_y, force_z]
            obj.maxpot                      = 0;                        % minimal    force state: [theta_torque, phi_torque, force_x, force_y, force_z]
            obj.minpot                      = 0;                        % maximal    force state: [theta_torque, phi_torque, force_x, force_y, force_z]            
            obj.it                          = 0;                        % iterator for accepted states. Different from "ittot", because NOT ALL states are calculated.
            obj.tr_fr_history               = NaN(1,5);                 % history of force states, matrix n*5: [theta_torque, phi_torque, force_x, force_y, force_z]

            % History of some parameters
            obj.xyz_th_phi_history          = NaN(1,5);                 % history of states x,y,z,theta, phi THAT WERE USED for force calculation. Matrix of n*5. 
            obj.XYZ                         = [];                       % FULL history of x,y,z.      Matrix of n*3: [x(:), y(:), z(:)], where n is current iteration number
            obj.ThPhi                       = [];                       % FULL history of theta, phi. Matrix of n*2: [theta(:), phi(:)], where n is current iteration number 
            obj.choix                       = 0;                        % FULL history of accepted/rejected states. If 1: state was favorable. 0 = rejected state (unfavorable), 2 = jump.

            % torque map and potential maps
            obj.angle_btw_mt_deg           = 10;                       % step of the MT azimuthal angles. used in force calc.

            obj.auto_calc_torque_map        = 0;                        % flag for auto calculation of the torque map
            obj.torque_map_angle_step       = 10;                       % angle step for the torque map
            obj.torque_map                  = [];                       % holder for the torque map
            
            
            % flags for automatic saving of the results
            obj.auto_save_nucleus_run       = 1;
            obj.auto_save_torque_map        = 1;
            obj.auto_save_state_history     = 1;
            obj.auto_save_split_image       = 1;

            obj.show_log                    = 1;
            obj.noLog_each_N_iteration      = 20; % if not logging: just show number of iteration (each N eterations).
            obj.show_visually               = 0;  
            obj.font_size                   = 14;
            obj.split_distance_threshold    = 1;
        end
        
        
      
        function set_cell(obj, current_cell_in)
            
             % abort if the cell has no shape
            if isempty(current_cell_in.shape_mat)
                disp(' ')
                disp('Cannot assign the cell: it has no shape')
                return;
            end
            
            % provided cell:
            obj.current_cell                     = current_cell_in;
            
            % new names for the files storing the results of this simulation
            obj.get_new_names;            
      
             % pixels belonging to the cell in each z-slice;
             % Example: XcYc{25} contains [x,y] of the 2D cell slice at z = 25.
            obj.XcYc                             = cell(size(obj.current_cell.shape_mat, 3), 1);                                                                                               
            for i = 1 : size(obj.current_cell.shape_mat, 3)
                [yc, xc] = find(obj.current_cell.shape_mat(:, :, i) == 1);
                obj.XcYc{i, 1} = [xc, yc];                              
            end
            
        end
        
        function get_new_names(obj)
            [s_fn, tm_in, nr_in, sh_in, sc_in]   = DPP_simulator.generate_sim_names(obj.current_cell);            
            obj.file_name_sim_mat                = s_fn;
            obj.file_name_torque_map_img         = tm_in;
            obj.file_name_nucleus_run_img        = nr_in;
            obj.file_name_state_history_img      = sh_in;       
            obj.file_name_split_cell_img         = sc_in;            
        end
        
        
        
        function reset(obj)
            disp(' ')
            disp('Resetting the iteration counter to 0.')
            disp('Resetting the "Equilibrium Found" flag to 0.')
            disp('ALL OTHER PARAMETERS STAY UNCHANGED.')
            disp('You may continue you simulation.')
            obj.cur_iter    = 0;
            obj.echec       = 0;
        end
        
        function cont_bool = continue_or_not(obj)
            
            cont_bool = 1;

            obj.check_if_equilibrium_configuration_is_reached;  % if eq. position is reached:  ECHEC -> 2.
            
            if obj.cur_iter >= obj.duration_full
                cont_bool = 0;
            end
            if obj.echec == 2 % echec = 2 equilibrium is found
                cont_bool = 0;
            end
            
        end
        
        
        
        function run(obj)
            % abort if no shape
            if isempty(obj.current_cell)
                disp('Cannot run simulation: the cell has no shape')
                return;
            end
            
            disp(' ')
            disp('Running Simulation')
            disp(' ')
            
            while obj.continue_or_not
                
                %  error check  
                % 0 is required to make the next iteration
                % 1 if delance was used (rethrowing) but nothing was changed.
                % 2 if equilibrium is reached
                % 3 if nucleus is outside the cell shape                
                % reset the error check from previous values
                obj.echec = 0; 
                
                obj.memorize_current_xyzthphi_state;            % save the current state. Changes "xsave, ysave, ..., phisave".
                obj.define_what_parameter_to_vary;              % find what parameter has to be varied. Changes "delance" and "re_roll_value"
                obj.vary_the_chosen_parameter;                  % Changes one of [phi,th,x,y,z,]. If nothing was changed: ECHEC -> 1.
                obj.check_if_nucleus_is_within_the_shape;       % check if the nucleus is inside If not: take old position and ECHEC -> 3

                if obj.echec == 0                               % ONLY if all checks have been passed (ECHEC = 0)
                    obj.cur_iter   = obj.cur_iter + 1;                % advance iteration for the new position
                    obj.calculate_forces_for_new_position;      % if it is inside: calculate the force; Updates several parameters!
                    obj.is_new_configuration_is_more_favorable;

                    %if ~obj.show_log                            % if not showing log, show iteration N.
                            if mod(obj.cur_iter, obj.noLog_each_N_iteration) == 0
                                disp(['     | iteration ' num2str(obj.cur_iter) '']);
                            end
                            
                            if obj.show_visually > 0
                                if mod(obj.cur_iter, obj.show_visually) == 0
                                    obj.current_cell.show
                                end
                            end
                    %end
                end
            end  
            
            % If failed to find the minimum: inform.
            if obj.cur_iter >= obj.duration_full && obj.echec <= 1  
                disp(' ')
                disp(['Finished ' num2str(obj.duration_full) ' iterations.'])
                disp(['MINIMUM HAS NOT BEEN FOUND.']);                
            elseif obj.echec == 2
                disp(' ')
                disp(['    Finishing the simulations: ']);
                disp(['    Nucleus x,y,z have been stable for indicated number of iterations: ' num2str(obj.duration_equilibrium)]);
            end
            
            
            %  ------------------    When finished    -------------------
            % (either when found equilibrium or run out of iterations)
            
            % save the final state of the simulation
            obj.save_simulation_state;
            
            % show the nucleus run
            obj.show_nucleus_run;
            
            % show the state history
            obj.show_state_history;

            % try to split the cell accordin the spindle orientation.
            % Ideally, should be done after proper theta and phi are set.
            obj.show_division_plane;

            
            % calculate and show the torque map
            if obj.auto_calc_torque_map                                 
                obj.torque_map_calculate;              
                % save the final stae of the simulation
                obj.save_simulation_state;                
                % show the torque map
                obj.show_torque_map;
            end      
        end 
        
        
        
        
        
        
        
        

        
        
        function check_if_equilibrium_configuration_is_reached(obj)
            % DEDEBUG:
            % this part is AP2016 style.
            % doesnt seem to work ideally (not in all cases)
            
            % check if the equilibrium state has been reached
            % and if the iterations should continue or not            
            % "obj.duration_equilibrium" is a number of iterations to consider a
            % position at equilibrium. If the parameters dont change for
            % this number of iterations, then the system is considered to
            % be equilibrated.            
            
            if obj.cur_iter > obj.duration_equilibrium    % if the number of iterations has reached the required amount
                
                % check if the position has not been changing more than +/- 1 pixel:
                % XYZ = [x(:), y(:), z(:)]; n*3 matrix where n = iteration number                
                x_old   = obj.XYZ(length(obj.choix) - obj.duration_equilibrium + 1, 1);
                x_last  = obj.XYZ(length(obj.choix), 1);
                y_old   = obj.XYZ(length(obj.choix) - obj.duration_equilibrium + 1, 2);
                y_last  = obj.XYZ(length(obj.choix), 2);
                z_old   = obj.XYZ(length(obj.choix) - obj.duration_equilibrium + 1, 3);
                z_last  = obj.XYZ(length(obj.choix), 3);
                
                if abs(x_old - x_last) <= 1 && ...
                   abs(y_old - y_last) <= 1 && ...
                   abs(z_old - z_last) <= 1 && ...          % if dx,dy,dz are minor;
                   obj.choix(end)      == 0 && ...          % and if the last movement was not accepted
                   sum(obj.tr_fr_state_current)   == sum(obj.minpot)   % and if the last configuration (tr_fr_state_current) is the most favorable (=obj.minpot). DEDEBUG : why like sum()???
                    
                    obj.echec          = 2;              	% eche ~= 0 => iteration has to stop: equilibrium is found.
                    obj.cur_iter          = obj.cur_iter + 1;   	% DEDEBUG : what for???                 
                end
            end            
        end
        
        
        
        
           
        function memorize_current_xyzthphi_state(obj)
            % Memorize the current positional parameters
            obj.xsave           = obj.current_cell.x;        
            obj.ysave           = obj.current_cell.y;
            obj.zsave           = obj.current_cell.z;
            obj.thetasave       = obj.current_cell.theta;
            obj.phisave         = obj.current_cell.phi;
        end
        
        
        
        
        
        
        function define_what_parameter_to_vary(obj)            
            % "delance": form french "re-throw"
            
            % this funciton generates "delance": probability of
            % re-thorwing a parameter "re_roll_value", which in turn determines what
            % parameter from (x,y,z,theta, phi) to vary.
            % if delance = 0, then no re-throwing for "re_roll_value" occurs and the
            % old varying parameter  (e.g. "x" or "theta") will be varied.

            % obj.choix - column of 0s and 1s. History of choices for 
            % taking/not taking a move: 1 = movement was accepted
            % (= favorable), 0 - not accepted (= not favorable).            
            
            if obj.choix(end) == 1                      % if the last movement was FAVORABLE:
                % Then there is a chance to continue varying the SAME parameter. 
                % THE SAME PARAMETER that was varied in the previous move and in
                % THE SAME DIRECTION (+1/-1) as in the previous move.
                % (one of the 5 parameters [x,y,z,th,phi])                
                obj.re_roll_bool = ceil(100 * rand(1, 1));  % choose randomly "delance" from (0,100)
                if obj.re_roll_bool <= obj.inertia          % compare to the "inertia" of the movement
                    obj.re_roll_bool = 2;                   % if inertia is stronger: NO RETHROW. [DEDEBUG: in AP2016 this var was set to 2 and not used anywhere]
                else
                    obj.re_roll_bool = 1;                   % if inertia is weaker: DO RETHROW. 
                end
                
            else
                obj.re_roll_bool = 1;                       % if a movement was NOT FAVORABLE, then DO RETHROW
            end   
            
            % If DO RETHROW:
            % choose randomly what parameter to vary and how.
            % 2 move increments (+1/-1) for 5 parameters ([x,y,z,th,phi])
            % this makes 10 random choices; lets make the range [1-10]:
            if obj.re_roll_bool == 1
                
                obj.re_roll_value  = ceil(10 * rand(1, 1)); 
                
                if obj.show_log
                    switch obj.re_roll_value
                        case 1
                            obj.message_str = ['     |  re-roll  | Vary theta +1 |' ];
                        case 2
                            obj.message_str = ['     |  re-roll  | Vary theta -1 |' ];
                        case 3
                            obj.message_str = ['     |  re-roll  | Vary phi   +1 |' ];
                        case 4
                            obj.message_str = ['     |  re-roll  | Vary phi   -1 |' ];
                        case 5
                            obj.message_str = ['     |  re-roll  | Vary x     +1 |' ];
                        case 6
                            obj.message_str = ['     |  re-roll  | Vary x     -1 |' ];
                        case 7
                            obj.message_str = ['     |  re-roll  | Vary y     +1 |' ];
                        case 8
                            obj.message_str = ['     |  re-roll  | Vary y     -1 |' ];
                        case 9
                            obj.message_str = ['     |  re-roll  | Vary z     +1 |' ];
                        case 10
                            obj.message_str = ['     |  re-roll  | Vary z     -1 |' ];

                        otherwise
                    end
                end
                
            else
                % no rethrow: keep curret "re_roll_value"
                % obj.re_roll_value = obj.re_roll_value;   
                if obj.show_log
                    obj.message_str = ['     |  inertia  | Vary the same |' ];
                end

            end            
                     
        end
        
        
        
        
        
        
     
        
        
        
        function vary_the_chosen_parameter(obj)
            % Apply the randomly chosen variation depending on the state of
            % "obj.re_roll_value" and previous state of "obj.tr_fr_state_current"
            % obj.re_roll_value  randomly varies from [1:10]
            
            
            % 4 WAYS FOR "theta" VARIATION
            if obj.re_roll_value == 1 && obj.current_cell.theta < 180 && obj.tr_fr_state_current(1) > 0
                if obj.show_log; disp(obj.message_str); end
                obj.thetasave               = obj.current_cell.theta;
                obj.current_cell.theta    	= obj.current_cell.theta + 1;
                obj.message_str = [sprintf('%-6s', 'theta '), '+1  | ', sprintf('%-6s', 'theta '), '= ', sprintf('%5d', obj.current_cell.theta) ' | '];
                
            elseif obj.re_roll_value == 1 && obj.current_cell.theta == 180 && obj.tr_fr_state_current(1) > 0
                if obj.show_log; disp(obj.message_str); end
                obj.thetasave               = obj.current_cell.theta;
                obj.current_cell.theta    	= 1; 
                obj.message_str = [sprintf('%-6s', 'theta '), '+1  | ', sprintf('%-6s', 'theta '), '= ', sprintf('%5d', obj.current_cell.theta) ' | '];
                
            elseif obj.re_roll_value == 2 && obj.current_cell.theta > 1 && obj.tr_fr_state_current(1) < 0
                if obj.show_log; disp(obj.message_str); end
                obj.thetasave               = obj.current_cell.theta;
                obj.current_cell.theta   	= obj.current_cell.theta - 1; 
                obj.message_str = [sprintf('%-6s', 'theta '), '-1  | ', sprintf('%-6s', 'theta '), '= ', sprintf('%5d', obj.current_cell.theta) ' | '];
                
            elseif obj.re_roll_value == 2 && obj.current_cell.theta == 1 && obj.tr_fr_state_current(1) < 0
                if obj.show_log; disp(obj.message_str); end
                obj.thetasave           	= obj.current_cell.theta;
                obj.current_cell.theta    	= 180; 
                obj.message_str = [sprintf('%-6s', 'theta '), '-1  | ', sprintf('%-6s', 'theta '), '= ', sprintf('%5d', obj.current_cell.theta) ' | '];
            
            % 4 WAYS FOR "phi" VARIATION
            elseif obj.re_roll_value == 3 && obj.current_cell.phi < 180 && obj.tr_fr_state_current(2) > 0
                if obj.show_log; disp(obj.message_str); end
                obj.phisave                 = obj.current_cell.phi;
                obj.current_cell.phi     	= obj.current_cell.phi + 1; 
                obj.message_str = [sprintf('%-6s', 'phi '), '+1  | ', sprintf('%-6s', 'phi '), '= ', sprintf('%5d', obj.current_cell.phi) ' | '];

            elseif obj.re_roll_value == 3 && obj.current_cell.phi == 180 && obj.tr_fr_state_current(2) > 0
                if obj.show_log; disp(obj.message_str); end
                obj.phisave                 = obj.current_cell.phi; 
                obj.thetasave               = obj.current_cell.theta; 
                obj.current_cell.theta      = 180 - obj.current_cell.theta;
                obj.current_cell.phi    	= 1; 
                obj.message_str = [sprintf('%-6s', 'phi '), '+1  | ', sprintf('%-6s', 'phi '), '= ', sprintf('%5d', obj.current_cell.phi) ' | '];
                
            elseif obj.re_roll_value == 4 && obj.current_cell.phi > 1 && obj.tr_fr_state_current(2) < 0
                if obj.show_log; disp(obj.message_str); end
                obj.phisave                 = obj.current_cell.phi;
                obj.current_cell.phi    	= obj.current_cell.phi - 1; 
                obj.message_str = [sprintf('%-6s', 'phi '), '-1  | ', sprintf('%-6s', 'phi '), '= ', sprintf('%5d', obj.current_cell.phi) ' | '];
                
            elseif obj.re_roll_value == 4 && obj.current_cell.phi == 1 && obj.tr_fr_state_current(2) < 0
                if obj.show_log; disp(obj.message_str); end
                obj.phisave                 = obj.current_cell.phi; 
                obj.thetasave               = obj.current_cell.theta;
                obj.current_cell.theta      = 180 - obj.current_cell.theta;
                obj.current_cell.phi        = 180; 
                obj.message_str = [sprintf('%-6s', 'phi '), '-1  | ', sprintf('%-6s', 'phi '), '= ', sprintf('%5d', obj.current_cell.phi) ' | '];
            
                
                
            % 2 WAYS FOR "x" VARIATION
            elseif obj.re_roll_value == 5 && ...
                   obj.tr_fr_state_current(3) > 0 && ...
                   ismember([obj.current_cell.x + 1, obj.current_cell.y], obj.XcYc{obj.current_cell.z, 1}, 'rows') 
                   % last condition: checks if after moving (x+1) 
                   % the resulting center's [x,y] is still inside 
                   % the shape slice
                if obj.show_log; disp(obj.message_str); end
                obj.xsave                   = obj.current_cell.x;
                obj.current_cell.x          = obj.current_cell.x + 1; 
                obj.message_str = [sprintf('%-6s', 'x '), '+1  | ', sprintf('%-6s', 'x '), '= ', sprintf('%5d', obj.current_cell.x) ' | '];

               
                
            elseif obj.re_roll_value == 6 && ...
                   obj.tr_fr_state_current(3) < 0 && ...
                   ismember([obj.current_cell.x - 1, obj.current_cell.y], obj.XcYc{obj.current_cell.z, 1}, 'rows')
                   % last condition: checks if after moving (x-1) 
                   % the resulting center's [x,y] is still inside 
                   % the shape slice
                if obj.show_log; disp(obj.message_str); end
                obj.xsave                   = obj.current_cell.x;
                obj.current_cell.x       	= obj.current_cell.x - 1;
                obj.message_str = [sprintf('%-6s', 'x '), '-1  | ', sprintf('%-6s', 'x '), '= ', sprintf('%5d', obj.current_cell.x) ' | '];
            
                
                
                
            % 2 WAYS FOR "y" VARIATION
            elseif obj.re_roll_value == 7 && ...
                   obj.tr_fr_state_current(4) > 0 && ...
                   ismember([obj.current_cell.x, obj.current_cell.y + 1], obj.XcYc{obj.current_cell.z, 1}, 'rows')
                   % last condition: checks if after moving (y+1) 
                   % the resulting center's [x,y] is still inside 
                   % the shape slice
                if obj.show_log; disp(obj.message_str); end
                obj.ysave                   = obj.current_cell.y;
                obj.current_cell.y      	= obj.current_cell.y + 1; 
                obj.message_str = [sprintf('%-6s', 'y '), '+1  | ', sprintf('%-6s', 'y '), '= ', sprintf('%5d', obj.current_cell.y) ' | '];

                
                
            elseif obj.re_roll_value == 8 && ...
                   obj.tr_fr_state_current(4) < 0 && ...
                   ismember([obj.current_cell.x, obj.current_cell.y - 1], obj.XcYc{obj.current_cell.z, 1}, 'rows') 
                   % last condition: checks if after moving (y-1) 
                   % the resulting center's [x,y] is still inside 
                   % the shape slice
                if obj.show_log; disp(obj.message_str); end
                obj.ysave                   = obj.current_cell.y;
                obj.current_cell.y        	= obj.current_cell.y - 1; 
                obj.message_str = [sprintf('%-6s', 'y '), '-1  | ', sprintf('%-6s', 'y '), '= ', sprintf('%5d', obj.current_cell.y) ' | '];
                
                
                
            % 2 WAYS FOR "z" VARIATION
            elseif obj.re_roll_value == 9 && ...
                   obj.tr_fr_state_current(5) > 0 && ...
                   obj.current_cell.z < size(obj.current_cell.shape_mat, 3)
               
               if ismember([obj.current_cell.x, obj.current_cell.y], obj.XcYc{obj.current_cell.z + 1, 1}, 'rows')
                    % last condition: checks if after moving (z+1) 
                   % the resulting center's [x,y] is still inside 
                   % at new shape slice (at z+1)
                    if obj.show_log; disp(obj.message_str); end
                    obj.zsave               = obj.current_cell.z;
                    obj.current_cell.z    	= obj.current_cell.z + 1;
                    obj.message_str = [sprintf('%-6s', 'z '), '+1  | ', sprintf('%-6s', 'z '), '= ', sprintf('%5d', obj.current_cell.z) ' | '];
               end
                
                
               
            elseif obj.re_roll_value == 10 && ...
                   obj.tr_fr_state_current(5) < 0 && ...
                   obj.current_cell.z > 1
               
                if ismember([obj.current_cell.x, obj.current_cell.y], obj.XcYc{obj.current_cell.z - 1, 1} ,'rows')
                    % last condition: checks if after moving (z+1) 
                   % the resulting center's [x,y] is still inside 
                   % at new shape slice (at z-1)
                    if obj.show_log; disp(obj.message_str); end
                    obj.zsave                   = obj.current_cell.z;
                    obj.current_cell.z       	= obj.current_cell.z - 1;
                    obj.message_str = [sprintf('%-6s', 'z '), '-1  | ', sprintf('%-6s', 'z '), '= ', sprintf('%5d', obj.current_cell.z) ' | '];
                end
             
               
            else
                % IF NEITHER OF WAYS FIRED OFF:
                % DEDEBUG: error?
                obj.echec = 1;
                obj.message_str = ''; % make it empty
                %['No condition for varying was met. re_roll_value = ' num2str(obj.de)];
            end
            
            
            
            if ~isempty(obj.message_str)
                iterator_format = ['%-', num2str(length(num2str(obj.duration_full))) 'd'];
                obj.message_str = [sprintf(iterator_format, obj.cur_iter), ' | ', obj.message_str, ' x, y, z, theta, phi = ['...
                    num2str(obj.current_cell.x), ', ', num2str(obj.current_cell.y), ', ', num2str(obj.current_cell.z), ', ' ...
                    num2str(obj.current_cell.theta), ', ', num2str(obj.current_cell.phi) ']'];
                
                if obj.show_log; disp(obj.message_str); end
            else
            end

        end
        
        
        
        
        
        
        
        
        
        
        function check_if_nucleus_is_within_the_shape(obj)            
            % check if all nucleus coordinates fit inside the cell shape
            
            
            if obj.current_cell.x == obj.xsave && ...
               obj.current_cell.y == obj.ysave && ...
               obj.current_cell.z == obj.zsave
                % if the position is same as old: 
                % do nothing.
                
                % disp(['    New position is the same...'])
                
            else                
                % if the position is new, check if the nucleus is still
                % inside  the cell:
                nucl_inside_bool = obj.current_cell.check_if_nucleus_is_within_the_shape;
                
                if ~nucl_inside_bool
                    obj.echec = 3;
                    
                    obj.message_str = [obj.message_str, ' | Nucleus out of the cell; Restoring the previous nucleus position.'];
                    if obj.show_log; disp(obj.message_str); end
                    obj.current_cell.x = obj.xsave;
                    obj.current_cell.y = obj.ysave;
                    obj.current_cell.z = obj.zsave;                    
                end
                % DEDEBUG: also AP2016 had: 
                % if a_nucleus_surf_point <0 or >size_shape_imgage:
                % echec = 3
             
            end
        end   
        
        
        
        
        
        
        
        
        
        
        function calculate_forces_for_new_position(obj)
            % Calculation of forces and torques for the given position and angles
                    
            % check first if current configuration exists already:
            xyz_th_ph               = obj.current_cell.get_state;            
            [ifexist, indice]       = ismember(xyz_th_ph, obj.xyz_th_phi_history,'rows');                

            if ifexist 
                
                obj.message_str = [obj.message_str, ' | Using previously calculated forces ...'];

                % if the configuration has already been calculated - reuse
                % the this calculation                       
                obj.tr_fr_state_new                        = obj.tr_fr_history(indice, :);
                % obj.tr_fr_history                     stays unchanged
                % obj.xyz_th_phi_history               stays unchanged
                
            else
                obj.message_str = [obj.message_str, ' | Calculating forces anew ...'];
                
                % calculate anew                
                obj.tr_fr_state_new                      =  force_calculator(obj);
                
                % Accumulate new results:
                obj.it                              = obj.it + 1;               % AP2106 style of concatenation...   
                obj.tr_fr_history(obj.it, :)            = obj.tr_fr_state_new;             % store calculated forces
                obj.xyz_th_phi_history(obj.it, 1)      = obj.current_cell.x;       % store calculated configuration
                obj.xyz_th_phi_history(obj.it, 2)      = obj.current_cell.y;
                obj.xyz_th_phi_history(obj.it, 3)      = obj.current_cell.z;
                obj.xyz_th_phi_history(obj.it, 4)      = obj.current_cell.theta;
                obj.xyz_th_phi_history(obj.it, 5)      = obj.current_cell.phi;  
            end
            
            if obj.show_log; disp(obj.message_str); end

        end
        
        
        
        
        
        function is_new_configuration_is_more_favorable(obj)            
            % THIS FUNCTION CHECKS IF NEW CONDITIONS ARE FAVORABLE
            
            % Also it checks whether there is a c hance for a "jump" EVEN if
            % the position is favorable. It is done because sometimes there
            % exist local ENERGY MINIMA, in which the the system can get
            % stuck. To avoid this, we introduce "jump" - it will kick out
            % the spindle from such a state, even if the state was
            % favorable.  
            % DEDEBUG: AP2016: jumping: keeping the new configuration anyways, to escape
            % from a local minima (might be due to pixelization or noise)
            
                
            de_saut     = 100 * rand(1, 1);                             % a throw for a jump [0:100].

            conda       = obj.init >= 1;                                % true if initialisation (for initialization init = 2)
            condb       = obj.init == 0 && (de_saut < obj.jump_chance); 	% DE DEBUG: AP2016 says "true if jumping"
            % According to AP then: 
            % chance for a jump = mean(100 * rand(1, 100000) < jump_chance) = jump_chance/100
            % if jump_chance = 1 then chance to jump is 0.01.
            % if (not initialization) and (a jump)
            % Then probability of condb is ~0.01 if "init" == 0.
            if (obj.re_roll_value == 1 || obj.re_roll_value == 2)
               % ''
            end
 
            abs_force_new   = sum(obj.tr_fr_state_new(3:5) .^ 2);
            abs_force_old   = sum(obj.tr_fr_state_current(3:5) .^ 2);
            
            % true if new THETA is OK
            conddtheta  = (obj.re_roll_value == 1 || obj.re_roll_value == 2) && (abs(obj.tr_fr_state_new(1)) < abs(obj.tr_fr_state_current(1))); 
            conddphi    = (obj.re_roll_value == 3 || obj.re_roll_value == 4) && (abs(obj.tr_fr_state_new(2)) < abs(obj.tr_fr_state_current(2)));
            condex      = (obj.re_roll_value == 5 || obj.re_roll_value == 6) && (abs_force_new  < abs_force_old);
            condey      = (obj.re_roll_value == 7 || obj.re_roll_value == 8) && (abs_force_new  < abs_force_old);
            condez      = (obj.re_roll_value == 9 || obj.re_roll_value == 10) && (abs_force_new  < abs_force_old);
            
%             conddtheta  = ((obj.re_roll_value == 1 || obj.re_roll_value == 2) && ...
%                            (sign(obj.tr_fr_state_current(1)) == sign(obj.tr_fr_state_new(1)) || ...
%                            (sign(obj.tr_fr_state_current(1)) ~= sign(obj.tr_fr_state_new(1)) && abs(obj.tr_fr_state_current(1)) > abs(obj.tr_fr_state_new(1))))); 

            % true if new PHI is OK
%             conddphi    = ((obj.re_roll_value == 3 || obj.re_roll_value == 4) && ...
%                            (sign(obj.tr_fr_state_current(2)) == sign(obj.tr_fr_state_new(2)) || ...
%                            (sign(obj.tr_fr_state_current(2)) ~= sign(obj.tr_fr_state_new(2)) && abs(obj.tr_fr_state_current(2)) > abs(obj.tr_fr_state_new(2)))));
%             
     
            % true if new X is OK            
%             condex      = ((obj.re_roll_value == 5 || obj.re_roll_value == 6) && ...
%                            (sign(obj.tr_fr_state_current(3)) == sign(obj.tr_fr_state_new(3)) || ...
%                            (sign(obj.tr_fr_state_current(3)) ~= sign(obj.tr_fr_state_new(3)) && abs(obj.tr_fr_state_current(3)) > abs(obj.tr_fr_state_new(3)))));
% 
%             % true if new Y is OK
%             condey      = ((obj.re_roll_value == 7 || obj.re_roll_value == 8) && ...
%                            (sign(obj.tr_fr_state_current(4)) == sign(obj.tr_fr_state_new(4)) || ...
%                            (sign(obj.tr_fr_state_current(4)) ~= sign(obj.tr_fr_state_new(4)) && abs(obj.tr_fr_state_current(4)) > abs(obj.tr_fr_state_new(4)))));
% 
%             % true if new Z is OK
%             condez      = ((obj.re_roll_value == 9 || obj.re_roll_value == 10) && ...
%                            (sign(obj.tr_fr_state_current(5)) == sign(obj.tr_fr_state_new(5)) || ...
%                            (sign(obj.tr_fr_state_current(5)) ~= sign(obj.tr_fr_state_new(5)) && abs(obj.tr_fr_state_current(5)) > abs(obj.tr_fr_state_new(5)))));
%             
                        
            % conditions to keep the new configuration : 
            % initialization || normal sim. run || more favorable (m.f.) phi || m.f. theta || m.f. x || m.f. y || m.f. z
            if conda || condb || conddphi || conddtheta || condex || condey || condez
                
                % one of the conditions was JUMP probability. We have to 
                % check separately if this is true:
                % if jump conditions is met
                % AND NOT the rest conditions ( = configuration IS FAVORABLE)
                if condb && ~(conddphi || conddtheta || condex || condey || condez)                    
                    obj.message_str = [obj.message_str, ' | JUMP! Configuration is LESS favorable, but it waccepted due to jump.'];
                    if obj.show_log; disp(obj.message_str); end
                    %disp('             Configuration is MORE favorable. However the system jumped out of it. --------- JUMP!');
                    obj.choix(end + 1) = 2;
                else
                    
                    % if the situation is MORE FAVORABLE: carry on.
                    obj.message_str = [obj.message_str, ' | Configuration is MORE favorable. This state is kept.'];
                    if obj.show_log; disp(obj.message_str); end
                    % disp('             Configuration is MORE favorable. This state is kept.')
                    obj.choix(end + 1, 1) = 1;
                    
                    % and update the value of the minimum potential
                    % it is supposed to be lower in a more favorable
                    % situation)
                    if sum(abs(obj.minpot)) > sum(abs(obj.tr_fr_state_new))
                        obj.minpot = obj.tr_fr_state_new;
                    end                    
                end

                % jumpp or not, memorize the new potential metric
                obj.tr_fr_state_current    = obj.tr_fr_state_new;
%                 obj.newmin      = [obj.current_cell.x, obj.current_cell.y, obj.current_cell.z,...
%                                    obj.current_cell.theta, obj.current_cell.phi];     
%                                

                % If we are only in the beginning of the simulations, initialize some lists:                
                if obj.init == 2      
                    obj.init        = 1;
                    obj.choix       =  obj.choix(2);        % reset the choice list ( the first element was 0)
                    obj.maxpot      =  obj.tr_fr_state_new;        % set the values for min and max potential metric
                    obj.minpot      =  obj.tr_fr_state_new;
                    
                elseif obj.init == 1
                    % if it is a second iteration, rearrange some values:
                    obj.init = 0;
                    if obj.tr_fr_state_new > obj.maxpot  % DEDEBUG: vector comparison!!!
                        obj.maxpot = obj.tr_fr_state_new;
                    else
                        obj.minpot = obj.tr_fr_state_new;
                    end
                    
                % DEDEBUG: if init == 0 : why dont we update the maxpot?
                % Should it be:
                % elseif obj.init == 0  
                %    if obj.tr_fr_state_new > obj.maxpot  % DEDEBUG: vector comparison!!!
                %        obj.maxpot = obj.tr_fr_state_new;
                %    else
                %        obj.minpot = obj.tr_fr_state_new;
                %    end
                    
                end

            else
                % if neither of conditions to accept the new positions are met:
                % it is not favorable or there is no jump:
                % do not take the new configuration, come back to previous
                obj.message_str = [obj.message_str, ' | Configuration is LESS favorable than previous. This state is rejected; restoring the previous configuration.'];
                if obj.show_log; disp(obj.message_str); end
                % disp('             Configuration is LESS favorable than previous. This state is rejected; restoring the previous configuration.');
                obj.choix(end + 1, 1) = 0;
                
                % Fall back onto a previous state.
                obj.current_cell.x           = obj.xsave;
                obj.current_cell.y           = obj.ysave;
                obj.current_cell.z           = obj.zsave;
                obj.current_cell.theta       = obj.thetasave;
                obj.current_cell.phi         = obj.phisave;
            end
                
            % record the history of the chosen [x,y,z] and [theta, phi]
            obj.XYZ     = [obj.XYZ;     [obj.current_cell.x, obj.current_cell.y, obj.current_cell.z]];     
            obj.ThPhi   = [obj.ThPhi;   [obj.current_cell.theta, obj.current_cell.phi]];            

            % disp(' ')
            % AP 2016 style:
            % obj.XYZ(size(obj.XYZ, 1) + 1, 1)    = obj.current_cell.x; 
            % obj.XYZ(size(obj.XYZ, 1), 2)        = obj.current_cell.y; 
            % obj.XYZ(size(obj.XYZ, 1), 3)        = obj.current_cell.z;                 
        end
        
        
        
        
    
        
        
        
        function torque_map_calculate(obj)
            disp(' ')
            disp(['Running torque map calculation.']) 

            % memorize the initial spindle orientation:
            theta_0                 = obj.current_cell.theta;
            phi_0                   = obj.current_cell.phi;                                   

            % Scan all the spindle orientations at a fixed nucleus position (x,y,z) unchanged.
            % Vary theta and phi only and calculate forces at a given orientation.
            obj.torque_map_theta    = 0 : obj.torque_map_angle_step : 180;
            obj.torque_map_phi      = 0 : obj.torque_map_angle_step : 180;
            
            TM_torque_theta         = zeros(length(obj.torque_map_phi), length(obj.torque_map_theta));
            TM_torque_phi           = zeros(length(obj.torque_map_phi), length(obj.torque_map_theta));

            
            for t = 1 : length(obj.torque_map_theta)
                
                cur_theta = obj.torque_map_phi(t);                
                disp(['    theta = ' num2str(cur_theta)])
                
                for p = 1 : length(obj.torque_map_phi)
                    
                    cur_phi = obj.torque_map_phi(p);                                    
                                       
                    % Set phi and theta to obj.current_cell.
                    % x,y,z unchanged (equilibrium position)
                    obj.current_cell.theta      = cur_theta; % theta
                    obj.current_cell.phi        = cur_phi;  % phi
                    
                    % Calculate the forces  for cvurrent orientation
                    torque_force            = force_calculator(obj);   % torque_force = [theta_torque, phi_torque, force_x_proj, force_y_proj, force_z_proj];
    
                    TM_torque_theta(p, t)  = torque_force(1);
                    TM_torque_phi(p, t)    = torque_force(2);
                end % phi loop
            end % theta loop
            
            obj.torque_map                  = (TM_torque_theta.^2 + TM_torque_phi .^ 2) .^ (1/2);
   

            % reset the initial spindle orientation
            obj.current_cell.theta         	= theta_0;
            obj.current_cell.phi          	= phi_0;
        end
        
        
        
        
        
        
         function save_simulation_state(obj)            
            
            disp(' ')
            disp('Saving the final state as:')
            disp(['    ', obj.file_name_sim_mat, '.mat']);              

            % figure handles shall be emptied; otherwise - large file:
            disp('    current cell: clearing axes handles...')
            current_simulation = obj;
            current_simulation.current_cell.clear_fig_handles;
            
            % saving
            disp(['    saving...'])
            save([obj.current_cell.cell_path, filesep, obj.file_name_sim_mat], 'current_simulation')
            disp(['    Saving Done.'])           
         end
        
         
         
         
         
         
         % =======================  "show" functions  =======================
         
          function show_state_history(obj)
            
            if isempty(obj.XYZ)
                disp(' ')
                disp('State history is empty.')
                return;
            else
                disp(' ')
                disp(['Showing state history.']) 
            end 
            
              
            iterations       = 1 : size(obj.XYZ, 1);
            [ind_acc, ~]     = find(obj.choix == 1);
            [ind_rej, ~]     = find(obj.choix == 0);
            [ind_jump, ~]    = find(obj.choix == 2);
            N_acc            = length(ind_acc);
            
            sz = get(0,'ScreenSize');
            figH = 0.8*sz(4);
            figW = figH;
            figY = 0.05*sz(4);
            figX = 0.1*sz(3);
            h0 = figure('position', [figX figY figW figH]);
                        
            subplot(3,1,1)
            set(gca, 'FontSize', obj.font_size);

            hold on;
            title({['History of State acception/rejection'],...
                    [obj.file_name_sim_mat]},...
                     'interpreter', 'none') % N accepted = ' num2str(N_acc)
            xlabel('iteration')
            ylabel('choice')
            ylim([-2 3])
            
            rej_color   = [1 0 0];
            acc_color   = [0 1 0];
            jump_color  = [1 1 0.5]; 
            
            legend_str  = {};            
            
            if ~isempty(ind_rej)
                plot(iterations(ind_rej), obj.choix(ind_rej), 'o', 'markerfacecolor', rej_color, 'markeredgecolor', 0.6*rej_color)
                legend_str = [legend_str, {'0 = rejected state'}];
            end
            if ~isempty(ind_acc)
                plot(iterations(ind_acc), obj.choix(ind_acc), 'o', 'markerfacecolor', acc_color, 'markeredgecolor', 0.6*acc_color)
                legend_str = [legend_str, {'1 = accepted state'}];
            end
            
            if obj.plot_jumps && ~isempty(ind_jump)
                plot(iterations(ind_jump), obj.choix(ind_jump),  'd', 'markerfacecolor', jump_color, 'markeredgecolor', 0.6*jump_color) 
                legend_str = [legend_str, {'2 = jump'}];            
            end
            
            legend(legend_str, 'location', 'northwest')


            
            
            subplot(3,1,2)
            set(gca, 'FontSize', obj.font_size);
            hold on; 
            title('From all iterations: evolution of [x,y,z, theta and phi]')
            xlabel('iteration')
            ylabel('pixel')
            
            theta_color = [1 0.5 1];
            phi_color   = [0.5 1 1];
            plot(iterations, obj.XYZ(:, 1), 'r.-')
            plot(iterations, obj.XYZ(:, 2), 'g.-')
            plot(iterations, obj.XYZ(:, 3), 'b.-')
            plot(iterations, obj.ThPhi(:, 1), 'o', 'markerfacecolor', theta_color, 'markeredgecolor', 0.6*theta_color, 'markersize', 4)
            plot(iterations, obj.ThPhi(:, 2), 'o', 'markerfacecolor', phi_color, 'markeredgecolor',     0.6*phi_color, 'markersize', 4)
            
            if obj.plot_jumps && ~isempty(ind_jump)
                plot(iterations(ind_jump), obj.ThPhi(ind_jump, 1), 'd', 'markerfacecolor', 0.5 * theta_color, 'markeredgecolor', 0.7 * theta_color, 'markersize', 6)
                plot(iterations(ind_jump), obj.ThPhi(ind_jump, 2), 'd', 'markerfacecolor', 0.5 * phi_color, 'markeredgecolor',   0.7 * phi_color, 'markersize', 6)
                plot(iterations(ind_jump), obj.XYZ(ind_jump, 1), 'rd')
                plot(iterations(ind_jump), obj.XYZ(ind_jump, 2), 'gd')
                plot(iterations(ind_jump), obj.XYZ(ind_jump, 3), 'bd')
                
                legend('x','y','z','theta', 'phi', 'jumps')
            else
                legend('x','y','z','theta', 'phi')
            end

          
            subplot(3,1,3)  
            set(gca, 'FontSize', obj.font_size);
            hold on;  
            % xlabel('iteration')
            % ylabel('pixel')
            title({['From force calculations: evolution of [x, y, z, theta, phi]']})
            
            % obj.xyz_th_phi_history = [x,y,z,theta, phi]
            plot(obj.xyz_th_phi_history(:, 1), 'r.-')
            plot(obj.xyz_th_phi_history(:, 2), 'g.-')
            plot(obj.xyz_th_phi_history(:, 3), 'b.-')
            plot(obj.xyz_th_phi_history(:, 4), 'd', 'markerfacecolor', theta_color, 'markeredgecolor', 0.6 * theta_color, 'markersize', 4)
            plot(obj.xyz_th_phi_history(:, 5), 'd', 'markerfacecolor', phi_color, 'markeredgecolor',   0.6 * phi_color, 'markersize', 4)
            
            legend('x','y','z', 'theta', 'phi')            
 
            if obj.auto_save_state_history
                disp(['    Saving as:']) 
                disp(['        ', obj.file_name_state_history_img, '.png'])
                
                try
                    print(h0, '-dpng', [ obj.current_cell.cell_path, filesep,  obj.file_name_state_history_img, '.png']);
                    disp('        Done')
                catch er
                    disp('        Could not save the image')
                    disp(['        ' er.message])
                end
            end
          end
        
          
          
          
          
          
          
          
          
        function show_nucleus_run(obj)
            
            if isempty(obj.XYZ)
                disp(' ')
                disp('Cell run is empty.')
                return;
            else
                disp(' ')
                disp(['Showing nucleus run.']) 
            end     
            
            % show the cell 
            obj.current_cell.show;  

            % and XYZ path.
            plot3(obj.XYZ(:,1), obj.XYZ(:, 2), obj.XYZ(:, 3), 'r.-')         
            
            xlabel('X', 'FontSize', 14)
            ylabel('Y', 'FontSize', 14)
            zlabel('Z', 'FontSize', 14)

            title({['Nucleus trajectory '],...
                   [obj.file_name_sim_mat]},...
                   'FontSize', 14, 'interpreter', 'none')

            if obj.auto_save_nucleus_run
                disp(['    Saving as:']) 
                disp(['        ', obj.file_name_nucleus_run_img, '.png'])
                try
                    print(gcf, '-dpng', [ obj.current_cell.cell_path, filesep,  obj.file_name_nucleus_run_img, '.png']);
                    disp('        Done')
                catch er
                    disp('        Could not save the image')
                    disp(['        ' er.message])
                end
            end
        end
        
        
        
        
        
        
        
        
        
        function show_torque_map(obj)
            
            if isempty(obj.torque_map)
                disp(' ')
                disp('Torque map is empty.')
                return
            else
                disp(' ')
                disp(['Showing torque map.']) 
            end
            
            sz = get(0,'ScreenSize');
            figH = 0.8*sz(4);
            figW = figH;
            figY = 0.05*sz(4);
            figX = 0.1*sz(3);
            h1 = figure('position', [figX figY figW figH]);
            
        
            [theta_mat, psi_mat]      = meshgrid(obj.torque_map_theta, obj.torque_map_phi);   
            surf(theta_mat, psi_mat, obj.torque_map, 'Facealpha', 0.6);
            
            axis tight;
            xlabel('theta', 'FontSize', 14)
            ylabel('phi', 'FontSize', 14)
            title({['Torque map'],...
                   [obj.file_name_sim_mat]},...
                   'FontSize', 14, 'interpreter', 'none')
            
            if obj.auto_save_torque_map
                disp(['    Saving as:']) 
                disp(['        ', obj.file_name_torque_map_img, '.png'])
               
                try
                    print(h1, '-dpng', [ obj.current_cell.cell_path, filesep,  obj.file_name_torque_map_img, '.png']);
                    
                    % save flat view:
                    view(2)
                    print(h1, '-dpng', [ obj.current_cell.cell_path, filesep,  obj.file_name_torque_map_img, '_top.png']);                    
                    disp('        Done')
                    
                catch er
                    disp('        Could not save the image')
                    disp(['        ' er.message])
                end
            end
                
        end
        
        
        
        
        
        
        
        
        
         function show_division_plane(obj)       
            disp(' ')
            err_bool = 0;
            
            try 
                disp('Trying to split the cell.')
            

                % spindle state:     
                x                       = obj.current_cell.x;
                y                       = obj.current_cell.y;
                z                       = obj.current_cell.z;

                mat_size              = size(obj.current_cell.shape_mat);

                % Division plane will be defined as a collection of points
                % that are not far from the spindle center, as measured ALONG the spindle.
                % For this we will calculate DOT PRODUCT of two vectors:
                % 1. spindle vector.
                % 2. any point vector
                % dot product will give the projection of the point-vector on
                % the spindle-vector, and we will select only those points that
                % are not far along this projection.           

                % 1. Vector parallel to the spindle:
                current_spindle_shape   = obj.current_cell.spindle_shape;

                spindle_vec             = [current_spindle_shape.x(2) - current_spindle_shape.x(1),...
                                           current_spindle_shape.y(2) - current_spindle_shape.y(1),...
                                           current_spindle_shape.z(2) - current_spindle_shape.z(1)];
                spindle_vec             = spindle_vec / norm(spindle_vec);

                % 2. Vector from the spindle position to a point within the cell
                % This vector I calculate for all point in the 3D image at once.
                % For this we have to generate all [x,y,z] coordinates of the
                % points in the 3D stack:            
                [X, Y, Z]               = meshgrid([1 : mat_size(2)],...
                                                   [1 : mat_size(1)],...
                                                   [1 : mat_size(3)]);
                % NOTE that X is the second dimension of the image, and Y is
                % the first. You may test that X is the second dimension with 
                % this snippet: you will see that indexing is in the correct order:
                if 0
                    a = rand(5,6,7);
                    [cols, rows, zs]               = meshgrid([1 : size(a, 2)], [1 : size(a, 1)], [1 : size(a, 3)]);
                    for i = 1: 10
                        [rows(i), cols(i), zs(i)] 
                    end
                end
                

                % Now 
                % X-x = x-components of all vectors, shifted by nucleus x
                % Y-y = y-components of all vectors, shifted by nucleus y
                % Z-z = z-components of all vectors, shifted by nucleus z
                % I mulptiply them by corresponding coordinate of the 
                % normal to the spindle and sum them up to get the dot product
                % for all vectors.            
                dot_prod_3Dmat              = (X - x) * spindle_vec(1) + ...
                                              (Y - y) * spindle_vec(2) + ...
                                              (Z - z) * spindle_vec(3);

                dot_prod_3Dmat(~logical(obj.current_cell.shape_mat))   	= 0;                    % if outside of the cell shape, set to 0.              
                dot_prod_3Dmat                                          = abs(dot_prod_3Dmat);  % we are interested in the absolute value of the dot products.
                dot_prod_3Dmat(dot_prod_3Dmat < obj.split_distance_threshold) 	= 0;        % if the distance is short enough: set image pix to 0
                dot_prod_3Dmat(dot_prod_3Dmat > obj.split_distance_threshold)	    = 1;        % if the distance is short enough: set image pix to 1
                dot_prod_3Dmat = uint8(dot_prod_3Dmat);


                % find connected 3D blobs
                connectivity_val            = 6;    % side-to-side connectivity for voxels
                % connectivity_val          = 18;   % side-to-side + rib-to-rib connectivity for voxels
                cc                          = bwconncomp(dot_prod_3Dmat, connectivity_val);  % list of conn components
                % there is often small junk after cutting . we will filter
                % them out below.
                
                if 0
                    % DEDEBUG: often when a body is cut, there are pixels hanging
                    % in air or connected with higher degree (e.g. 18 and
                    % so on). One can see them with this snippet:
                    template_uint8_mat           = uint8(zeros(mat_size));
                    all_bodies = {};
                    figure;
                    hold on;
                    axis equal;
                    xlim([1, mat_size(2)]);
                    ylim([1, mat_size(1)]);
                    zlim([1, mat_size(3)]);
                    xlabel('x')
                    ylabel('y')
                    zlabel('z')
                    for i = 1 : cc.NumObjects
                        if length(cc.PixelIdxList{i})<100
                            ggg         = template_uint8_mat;
                            ggg(cc.PixelIdxList{i}) = 1;
                            %all_bodies = [all_bodies, {ggg}];
                            patch(isosurface(ggg, 0.5),'facecolor',rand(1,3),'edgecolor', 'none', 'facealpha', 1)
                        end
                    end                   
                 end
                
                 
                % filter out junk and take two largest bodies.
                bb                          = regionprops(cc,'Area');
                volumes                     = cat(1, bb.Area);
                
                % sort and take the last two (the largest two).
                [~, sorted_inds]            = sort(volumes);
                sorted_inds                 = sorted_inds(end-1 : end); 
                
                two_bodies_pixels_idx       = cc.PixelIdxList(sorted_inds);
                                
                volume_ratio                = min(volumes(sorted_inds)) / ...
                                              max(volumes(sorted_inds));
                
                template_uint8_mat           = uint8(zeros(mat_size));
                daughter_cells_uint8_mats    = {};
                for j   = 1 : length(two_bodies_pixels_idx)
                    pix_idx                         = two_bodies_pixels_idx{j};
                    cur_3Dmat                       = template_uint8_mat;
                    cur_3Dmat(pix_idx)              = 1;
                    daughter_cells_uint8_mats{j}    = cur_3Dmat;
                end
               
            catch err                
                disp('Could not split the cell.')
                disp(err.message)
                err_bool = 1;
            end
            
            
            if ~err_bool

                col_set = [1, 0.5, 0;...
                           0, 1, 1]; % orange; cyan.

            sz = get(0,'ScreenSize');
            figH = 0.8*sz(4);
            figW = figH;
            figY = 0.05*sz(4);
            figX = 0.1*sz(3);
            h2 = figure('position', [figX figY figW figH]);
            
            hold on
                axis equal
                xlabel('X', 'FontSize', obj.font_size);
                ylabel('Y', 'FontSize', obj.font_size);
                zlabel('Z', 'FontSize', obj.font_size);
                xlim([0 mat_size(2)]);
                ylim([0 mat_size(1)]);
                zlim([0 mat_size(3)]);
                view([25 25])
                box on;

                 title({['Division plane according to the spindle position'],...
                        [obj.file_name_sim_mat], ...
                        ['Volume Small / Volume Large: ' num2str(volume_ratio)] }, ...
                        'FontSize', obj.font_size, 'interpreter', 'none')

                for j = 1 : 2
                    patch(isosurface(daughter_cells_uint8_mats{j}, 0.5), ...
                            'FaceColor', col_set(j, :), 'FaceAlpha', 0.15, 'EdgeColor', 'none', 'EdgeAlpha', 0.5);
                end

                 % nucleus surface as excluded volume
                 nucl_patch = surf2patch(obj.current_cell.nucleus_shape_excl(:, :, 1), ...
                                         obj.current_cell.nucleus_shape_excl(:, :, 2), ...
                                         obj.current_cell.nucleus_shape_excl(:, :, 3));
                 patch(nucl_patch, 'FaceColor', obj.current_cell.col_nucleus_surf, 'FaceAlpha',0.05, 'EdgeColor','none');

                 % nucleus surface as normal volume
                 surf2patch(obj.current_cell.nucleus_shape.x, ...
                            obj.current_cell.nucleus_shape.y, ...
                            obj.current_cell.nucleus_shape.z);                    
                 patch(nucl_patch, 'FaceColor', obj.current_cell.col_nucleus_surf, 'FaceAlpha',0.05, 'EdgeColor','none');

                 % nucleus center
                 plot3(obj.current_cell.x, obj.current_cell.y, obj.current_cell.z, ...
                        'o', 'markerfacecolor', obj.current_cell.col_nucleus_center, ...
                        'markeredgecolor', 0.6 * obj.current_cell.col_nucleus_center)

                 % spindle.
                 % recalculate the spindle chape
                 current_spindle_shape           = obj.current_cell.spindle_shape;
                 % set the data
                 plot3(current_spindle_shape.x, ...
                       current_spindle_shape.y, ...
                       current_spindle_shape.z, '-', 'color', obj.current_cell.col_spindle)

                if obj.auto_save_split_image
                    disp(['    Saving as:']) 
                    disp(['        ', obj.file_name_split_cell_img, '.png'])
                    try 
                        print(h2, '-dpng', [ obj.current_cell.cell_path, filesep,  obj.file_name_split_cell_img, '.png']);
                        disp('        Done')
                    catch er
                        disp('        Could not save the image')
                        disp(['        ' er.message])
                    end
                end
            end
        end
        
    end  
     
    
    
            
    methods (Static)


        function [file_name_sim_mat, file_name_torque_map,  file_name_nucl_run, file_name_state_history, file_name_split_cell] = generate_sim_names(current_cell)
           
            
            % first try to find if a "name_1.mat" file already exists
            root_path           = current_cell.cell_path;
            cur_ind             = 1;            
            prefix_generator    = @(n) ['run_', num2str(n)];
            
            % if exists, update it to name_2, name_3 etc until a non-existing name is found.
            while exist([root_path, prefix_generator(cur_ind), '_simulation.mat'], 'file') 
                cur_ind = cur_ind + 1;
            end
            
            file_name_sim_mat           = [prefix_generator(cur_ind), '_simulation'];     % name of the simulation save file (random walk history)
            file_name_torque_map        = [prefix_generator(cur_ind), '_torque_map'];              % name of the torque map img file
            file_name_nucl_run          = [prefix_generator(cur_ind), '_nucleus_run'];             % name pattern for the nucleus run img
            file_name_state_history     = [prefix_generator(cur_ind), '_state_history'];           % name pattern for the state history  img
            file_name_split_cell        = [prefix_generator(cur_ind), '_split_cell'];              % name pattern for the split cell  img
            
            
            disp(' ')
            disp('File names for the current simulation:')
            disp(['    ' root_path])
            disp(['    ' file_name_sim_mat]); 
            disp(['    ' file_name_torque_map]);     
            disp(['    ' file_name_nucl_run]); 
            disp(['    ' file_name_state_history]); 
            disp(['    ' file_name_split_cell]);

        end

    end
    
    
    
    
    
end